<?php
/**
 * Framework functions
 *
 * @package quper
 * @version 3.6.8
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get Settings options of elementor
function quper_elementor( $settings ) {
	if ( ! class_exists( '\Elementor\Plugin' ) ) { return false; }

	// Get the current post id
	$post_id = get_the_ID();

	// Get the page settings manager
	$page_settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers( 'page' );

	// Get the settings model for current post
	$page_settings_model = $page_settings_manager->get_model( $post_id );

	return  $page_settings_model->get_settings( $settings );

}

// Return class for reploader site
function quper_preloader_class() {
	// Get page preloader option from theme mod
	$class = quper_get_mod( 'preloader', 'animsition' );
	return esc_attr( $class );
}

// Get layout position for pages
function quper_layout_position() {
	// Default layout position
	$layout = 'sidebar-right';

	// Get layout position for site
	$layout = quper_get_mod( 'site_layout_position', 'sidebar-right' );

	// Get layout position for page blog
	if ( is_page() )
		$layout = quper_get_mod( 'custom_page_layout_position', 'no-sidebar' );

	// Get layout position for single post
	if ( is_singular( 'post' ) )
		$layout = quper_get_mod( 'single_post_layout_position', 'sidebar-right' );

	// Get layout position for shop pages
	if ( class_exists( 'woocommerce' ) ) {
		if ( is_shop() || is_product_category() )
			$layout = quper_get_mod( 'shop_layout_position', 'no-sidebar' );  
		if ( is_singular( 'product' ) )
			$layout = quper_get_mod( 'shop_single_layout_position', 'no-sidebar' );
		if ( is_cart() || is_checkout() ) {
			$layout = 'no-sidebar';
		}
	}

	// Other single except single post
	if ( is_single() && !is_singular( 'post' ) ) 
		$layout = 'no-sidebar';

	// Get layout position for single project
	if ( is_singular( 'project' ) )
		$layout = quper_get_mod( 'single_project_layout_position', 'no-sidebar' );

	// Get layout position for single service
	if ( is_singular( 'service' ) )
		$layout = quper_get_mod( 'single_service_layout_position', 'no-sidebar' );

	// Elementor settings
	if ( quper_elementor( 'site_layout_position' ) ) {
		$layout = quper_elementor( 'site_layout_position' );
	} 

	return $layout;
}

// Theme pagination
function quper_pagination( $query = '', $echo = true ) {
	
	$prev_arrow = '<i class="elegant-arrow_carrot-left"></i>';
	$next_arrow = '<i class="elegant-arrow_carrot-right"></i>';

	if ( ! $query ) {
		global $wp_query;
		$query = $wp_query;
	}

	$total  = $query->max_num_pages;
	$big    = 999999999;

	// Display pagination
	if ( $total > 1 ) {

		// Get current page
		if ( $current_page = get_query_var( 'paged' ) ) {
			$current_page = $current_page;
		} elseif ( $current_page = get_query_var( 'page' ) ) {
			$current_page = $current_page;
		} else {
			$current_page = 1;
		}

		// Get permalink structure
		if ( get_option( 'permalink_structure' ) ) {
			if ( is_page() ) {
				$format = 'page/%#%/';
			} else {
				$format = '/%#%/';
			}
		} else {
			$format = '&paged=%#%';
		}

		$args = array(
			'base'      => str_replace( $big, '%#%', html_entity_decode( get_pagenum_link( $big ) ) ),
			'format'    => $format,
			'current'   => max( 1, $current_page ),
			'total'     => $total,
			'mid_size'  => 3,
			'type'      => 'list',
			'prev_text' => $prev_arrow,
			'next_text' => $next_arrow
		);

		// Output
		if ( $echo ) {
			echo '<div class="quper-pagination clearfix">'. paginate_links( $args ) .'</div>';
		} else {
			return '<div class="quper-pagination clearfix">'. paginate_links( $args ) .'</div>';
		}

	}
}

// Render blog entry blocks
function quper_blog_entry_layout_blocks() {

	// Get layout blocks
	$blocks = quper_get_mod( 'blog_entry_composer' );

	// If blocks are 100% empty return defaults
	$blocks = $blocks ? $blocks : 'meta,title,excerpt_content,readmore';

	// Convert blocks to array so we can loop through them
	if ( ! is_array( $blocks ) ) {
		$blocks = explode( ',', $blocks );
	}

	// Set block keys equal to vals
	$blocks = array_combine( $blocks, $blocks );

	// Return blocks
	return $blocks;
}

// Render blog meta items
function quper_entry_meta() {
	// Get meta items from theme mod
	$meta_item = quper_get_mod( 'blog_entry_meta_items', array( 'date', 'categories' ) );

	// If blocks are 100% empty return defaults
	$meta_item = $meta_item ? $meta_item : 'author,comments';

	// Turn into array if string
	if ( $meta_item && ! is_array( $meta_item ) ) {
		$meta_item = explode( ',', $meta_item );
	}

	// Set keys equal to values
	$meta_item = array_combine( $meta_item, $meta_item );

	// Loop through items
	foreach ( $meta_item as $item ) :
		if ( 'author' == $item ) {
			printf( '<span class="post-by-author item">%4$s <a class="name" href="%1$s" title="%2$s">%3$s</a></span>',
				esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
				esc_attr( sprintf( esc_html__( 'View all posts by %s', 'quper' ), get_the_author() ) ),
				get_the_author(),
				esc_html__( 'by', 'quper' )
			);
			
		}
		else if ( 'comments' == $item ) {
			if ( comments_open() || get_comments_number() ) {
				echo '<span class="post-comment item"><span class="inner">';
				comments_popup_link( esc_html__( '0 comments', 'quper' ), esc_html__( '1 Comment', 'quper' ), esc_html__( '% Comments', 'quper' ) );
				echo '</span></span>';
			}
		}
		else if ( 'date' == $item ) {
			printf( '<span class="post-date item"><span class="entry-date">%1$s</span></span>',
				get_the_date()
			);
		}
		else if ( 'categories' == $item ) {
			echo '<span class="post-meta-categories item">' . esc_html__( 'in ', 'quper' );
			the_category( ', ', get_the_ID() );
			echo '</span>';
		}
	endforeach;
}

// Return background CSS
function quper_bg_css( $style ) {
	$css = '';
	if ( $style = quper_get_mod( $style ) ) {
		if ( 'fixed' == $style ) {
			$css .= ' background-position: center center; background-repeat: no-repeat; background-attachment: fixed; background-size: cover;';
		} elseif ( 'fixed-top' == $style ) {
			$css .= ' background-position: center top; background-repeat: no-repeat; background-attachment: fixed; background-size: cover;';
		} elseif ( 'fixed-bottom' == $style ) {
			$css .= ' background-position: center bottom; background-repeat: no-repeat; background-attachment: fixed; background-size: cover;';
		} elseif ( 'cover' == $style ) {
			$css .= ' background-repeat: no-repeat; background-position: center top; background-size: cover;';
		} elseif ( 'center-top' == $style ) {
			$css .= ' background-repeat: no-repeat; background-position: center top;';
		} elseif ( 'repeat' == $style ) {
			$css .= ' background-repeat: repeat;';
		} elseif ( 'repeat-x' == $style ) {
			$css .= ' background-repeat: repeat-x;';
		} elseif ( 'repeat-y' == $style ) {
			$css .= ' background-repeat: repeat-y;';
		}
	}

	return esc_attr( $css );
}

// Return background css for elements
function quper_element_bg_css( $bg ) {
	$css = '';
	$style = $bg .'_style';

	if ( $bg_img = quper_get_mod( $bg ) )
		$css .= 'background-image: url('. esc_url( $bg_img ). ');';

	$css .= quper_bg_css( $style );

	return esc_attr( $css );
}

// Return background css for featured title area
function quper_featured_title_bg() {
	$css = '';
	
	if ( is_page() ) {
		$page_bg_url = '';
		$bg_img = quper_get_mod( 'featured_title_background_img' );
		if ( !$page_bg_url && $bg_img ) {
			$css .= 'background-image: url('. esc_url( $bg_img ) .');';
		} else {
			$css .= 'background-image: url('. esc_url( $page_bg_url ) .');';
		}
		
	} elseif ( is_single() && ( $bg_img = quper_get_mod( 'blog_single_featured_title_background_img' ) ) ) {
		$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	} elseif ( $bg_img = quper_get_mod( 'featured_title_background_img' ) ) {
		$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	}

	if ( quper_is_woocommerce_shop() && $bg_img = quper_get_mod( 'shop_featured_title_background_img' ) ) {
		$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	}

	if ( is_singular( 'product' ) && $bg_img = quper_get_mod( 'shop_single_featured_title_background_img' ) ) {
		$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	}

	if ( is_tax() || is_singular( 'project' ) ) {
		if ( $bg_img = quper_get_mod( 'project_single_featured_title_background_img' ) )
			$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	}

	$css .= quper_bg_css('featured_title_background_img_style');

	return esc_attr( $css );
}

// Return background for main content area
function quper_main_content_bg() {
	$css = '';

	if ( $bg_img = quper_get_mod( 'main_content_background_img' ) ) {
		$css = 'background-image: url('. esc_url( $bg_img ). ');';
	}

	if ( is_page() ) {
		if ( quper_metabox('main_content_bg') ) {
			$css = 'background-color:'. quper_metabox('main_content_bg') .';';
		}
		if ( quper_metabox('main_content_bg_img') ) {
			$images = quper_metabox( 'main_content_bg_img', array( 'size' => 'full', 'limit' => 1 ) );
			$image = reset( $images );
			$css = 'background-image: url('. esc_url( $image['url'] ) .');';
		}
	}

	$css .= quper_bg_css('main_content_background_img_style');

	return esc_attr( $css );
}

add_action( 'after_setup_theme', 'quper_main_content_bg' );

// Return background for footer area
function quper_footer_bg() {
	$css = '';

	if ( is_page() ) {
		if ( quper_metabox('footer_bg') ) {
			$css .= 'background-color:'. quper_metabox('footer_bg') .';';
		}
		if ( quper_metabox('footer_bg_img') ) {
			$images = quper_metabox( 'footer_bg_img', array( 'size' => 'full', 'limit' => 1 ) );
			$image = reset( $images );
			$css .= 'background-image: url('. esc_url( $image['url'] ) .');';
		}
	} elseif ( $bg_img = quper_get_mod( 'footer_bg_img' ) ) {
		$css .= 'background-image: url('. esc_url( $bg_img ) .');';
	}

	$css .= quper_bg_css('footer_bg_img_style');

	return esc_attr( $css );
}

// Returns array of social
function quper_header_social_options() {
	return apply_filters ( 'quper_header_social_options', array(
		'facebook' => array(
			'label' => esc_html__( 'Facebook', 'quper' ),
			'icon_class' => 'fab fa-facebook-f',
		),
		'twitter' => array(
			'label' => esc_html__( 'Twitter', 'quper' ),
			'icon_class' => 'fab fa-twitter',
		),
		'instagram'  => array(
			'label' => esc_html__( 'Instagram', 'quper' ),
			'icon_class' => 'fab fa-instagram',
		),
		'youtube' => array(
			'label' => esc_html__( 'Youtube', 'quper' ),
			'icon_class' => 'fab fa-youtube',
		),
		'dribbble'  => array(
			'label' => esc_html__( 'Dribbble', 'quper' ),
			'icon_class' => 'fab fa-dribbble',
		),
		'vimeo' => array(
			'label' => esc_html__( 'Vimeo', 'quper' ),
			'icon_class' => 'fab fa-vimeo',
		),
		'tumblr'  => array(
			'label' => esc_html__( 'Tumblr', 'quper' ),
			'icon_class' => 'fab fa-tumblr',
		),
		'pinterest'  => array(
			'label' => esc_html__( 'Pinterest', 'quper' ),
			'icon_class' => 'fab fa-pinterest',
		),
		'linkedin'  => array(
			'label' => esc_html__( 'LinkedIn', 'quper' ),
			'icon_class' => 'fab fa-linkedin',
		),
	) );
}

// Check if it is WooCommerce Pages
function quper_is_woocommerce_page() {
    if ( function_exists ( "is_woocommerce" ) && is_woocommerce() )
		return true;

    $woocommerce_keys = array (
    	"woocommerce_shop_page_id" ,
        "woocommerce_terms_page_id" ,
        "woocommerce_cart_page_id" ,
        "woocommerce_checkout_page_id" ,
        "woocommerce_pay_page_id" ,
        "woocommerce_thanks_page_id" ,
        "woocommerce_myaccount_page_id" ,
        "woocommerce_edit_address_page_id" ,
        "woocommerce_view_order_page_id" ,
        "woocommerce_change_password_page_id" ,
        "woocommerce_logout_page_id" ,
        "woocommerce_lost_password_page_id" );

    foreach ( $woocommerce_keys as $wc_page_id ) {
		if ( get_the_ID () == get_option ( $wc_page_id , 0 ) ) {
			return true ;
		}
    }
    
    return false;
}

// Checks if is WooCommerce Shop page
function quper_is_woocommerce_shop() {
	if ( ! class_exists( 'woocommerce' ) ) {
		return false;
	} elseif ( is_shop() ) {
		return true;
	}
}

// Checks if is WooCommerce archive product page
function quper_is_woocommerce_archive_product() {
	if ( ! class_exists( 'woocommerce' ) ) {
		return false;
	} elseif ( is_product_category() || is_product_tag() ) {
		return true;
	}
}

// Returns correct ID for any object
function quper_parse_obj_id( $id = '', $type = 'page' ) {
	if ( $id && function_exists( 'icl_object_id' ) ) {
		$id = icl_object_id( $id, $type );
	}
	return $id;
}

// Hexdec color string to rgb(a) string
function quper_hex2rgba( $color, $opacity = false ) {
 	$default = 'rgb(0,0,0)';

	if ( empty( $color ) ) return $default; 
    if ( $color[0] == '#' ) $color = substr( $color, 1 );

    if ( strlen( $color ) == 6 ) {
		$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
    } elseif ( strlen( $color ) == 3 ) {
        $hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
    } else {
        return $default;
    }

    $rgb =  array_map( 'hexdec', $hex );

    if ( $opacity ) {
    	if ( abs($opacity ) > 1 ) $opacity = 1.0;
    	$output = 'rgba('. implode( ",", $rgb ) .','. $opacity .')';
    } else {
    	$output = 'rgb('. implode( ",", $rgb ) .')';
    }

    return $output;
}

// SVG Core Icons
function quper_svg( $icon ) {
	$svg = array(
		'cart' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
						<path d="M20.4,16.8H4.8c-0.7,0-1.2-0.5-1.2-1.2V2.4H0V0h4.8C5.5,0,6,0.5,6,1.2v13.2h13.5l1.8-7.2H8.4V4.8
	h14.4C23.5,4.8,24,5.3,24,6c0,0.1,0,0.2,0,0.3l-2.4,9.6C21.4,16.4,21,16.8,20.4,16.8z M3.6,21.6c0-1.3,1.1-2.4,2.4-2.4
	c1.3,0,2.4,1.1,2.4,2.4S7.3,24,6,24S3.6,22.9,3.6,21.6C3.6,21.6,3.6,21.6,3.6,21.6z M16.8,21.6c0-1.3,1.1-2.4,2.4-2.4
	c1.3,0,2.4,1.1,2.4,2.4S20.5,24,19.2,24C17.9,24,16.8,22.9,16.8,21.6L16.8,21.6z"/>
					</svg>',
		'search' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
						<path d="M9.3,18.7C4.2,18.7,0,14.5,0,9.3C0,4.2,4.2,0,9.3,0c5.2,0,9.3,4.2,9.3,9.3
	C18.7,14.5,14.5,18.7,9.3,18.7z M9.3,2.7c-3.7,0-6.7,3-6.7,6.7s3,6.7,6.7,6.7C13,16,16,13,16,9.3C16,5.6,13,2.7,9.3,2.7z M21.7,23.6
	l-4-4c-0.5-0.5-0.5-1.4,0-1.9c0.5-0.5,1.3-0.5,1.8,0l4,4c0.5,0.5,0.5,1.4,0,1.9C23.1,24.1,22.3,24.2,21.7,23.6
	C21.7,23.6,21.7,23.6,21.7,23.6z"/>
					</svg>',
		'menu' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
						<path d="M28.5,4.5h-27C0.7,4.5,0,3.8,0,3s0.7-1.5,1.5-1.5h27C29.3,1.5,30,2.2,30,3S29.3,4.5,28.5,4.5z
	 M15,13.5H1.5C0.7,13.5,0,12.8,0,12s0.7-1.5,1.5-1.5H15c0.8,0,1.5,0.7,1.5,1.5S15.8,13.5,15,13.5z M28.5,22.5h-27
	C0.7,22.5,0,21.8,0,21s0.7-1.5,1.5-1.5h27c0.8,0,1.5,0.7,1.5,1.5S29.3,22.5,28.5,22.5z"/>
					</svg>',
		'arrow_left' => '',
		'arrow_right' => '',
	);

	if ( array_key_exists( $icon, $svg) ) {
		return $svg[$icon];
	} else {
		return null;
	}
}

// Get All Pages
function quper_get_pages() {
	$args = [
        'post_type' => 'page',
        'posts_per_page' => -1,
    ];

    $pages = get_posts($args);
    $options = [];

    if (!empty($pages) && !is_wp_error($pages)) {
        foreach ($pages as $page) {
            $options[$page->ID] = $page->post_title;
        }
    }
    return $options;
}
